﻿Public Class RADecToAltAz

    ' Convert RA and Dec co-ordinates to Altitude and Azimuth for a specified observers location and local time.
    '
    ' One public facing function in this class: RADecToAltAz()
    '
    ' Keith Ehren
    ' www.astroworkbench.co.uk

    ReadOnly util As New utils  ' Class containing utility functions

    ' Entry function for converting RA and Dec to Altitude and Azimuth for the specified local date, time,
    ' Daylight Saving, Time Zone and the observers location.
    '
    ' This function returns Altitude, Azimuth, UT, GMST and LST
    Public Sub RADecToAltAz(ByVal pRA As Double,                 ' RA
                                 ByVal pDec As Double,           ' Dec           
                                 ByVal pLongitude As Double,    ' Observers longitude in degrees
                                 ByVal pLatitude As Double,     ' Observers latitude in Radians
                                 ByVal pDy As Long,             ' Day
                                 ByVal pMn As Long,             ' Month
                                 ByVal pYr As Long,             ' Year
                                 ByVal pTm As Double,           ' Time
                                 ByVal pDS As Integer,          ' Daylight saving
                                 ByVal pTZ As Integer,          ' Time zone
                                 ByRef pUT As Double,           ' Return Universal Time
                                 ByRef pGST As Double, ' Return Greenwich Siderial Time
                                 ByRef pLST As Double, ' Return Local Siderial Time
                                 ByRef pAlt As Double, ' Return Altitude
                                 ByRef pAz As Double)  ' Return Azimuth

        Try


            Dim lAzEquatorial As Double ' Equatorial is based upon a fixed position at earth centre and celestial poles
            Dim lDecEquatorial As Double
            Dim lAzHorizontal As Double
            Dim lAltHorizontal As Double


            convertLocalTimeToLST(pDS, pTZ, pLongitude, pDy, pMn, pYr, pTm, pUT, pGST, pLST)

            Dim lHrAngle As Double = hourAngle(pRA, pGST, pLongitude)

            lDecEquatorial = util.degreesToRads(pDec)
            lAzEquatorial = util.degreesToRads(lHrAngle * 15.0)

            equatorialToHorizontal(lAzEquatorial, lDecEquatorial, pLatitude, lAzHorizontal, lAltHorizontal)

            pAz = util.radsToDegrees(lAzHorizontal)
            pAlt = util.radsToDegrees(lAltHorizontal)


        Catch ex As Exception
            util.Err(ex.Message, System.Reflection.MethodBase.GetCurrentMethod.Name)
        End Try
    End Sub




    ' Convert local time to UT (Univeral Time), GST (Greenwich Siderial Time) and LST (Local Siderial Time)
    Private Sub convertLocalTimeToLST(ByVal pDs As Integer, ' Daylight saving
                                            ByVal pTz As Integer,       ' Time zone (+ve or -ve)
                                            ByVal pLongitude As Double, ' Observers longitude in degrees
                                            ByVal pDy As Integer,      ' Local day
                                            ByVal pMn As Integer,      ' Local month
                                            ByVal pYr As Integer,      ' Local year
                                            ByVal pLocalTm As Double,  ' Local time
                                            ByRef pUT As Double,       ' Return UT
                                            ByRef pGST As Double,      ' Return GST
                                            ByRef pLST As Double)      ' return LST


        Try


            Dim lNxtPrvDayIndicator As Integer
            Dim lT As Double
            Dim lR0 As Double
            Dim lR1 As Double
            Dim lT0 As Double


            ' A lot of magic numbers in the calculation below, but as you would assume, they all have significant meaning.
            ' For example, 0.0657098244 and 1.002737908 are from the formula relating to Greenwich mean sidereal time (GMST):
            '  GMST = G + 0.0657098244 × d + 1.00273791 × ut
            ' Where:
            ' - G Is a constant that varies for each year and was the Greenwich hour angle of the Sun at a specified Epoch (epoch 2000 used below).
            ' - d is the day number of the year
            ' - ut is the universal time
            ' 1.002737908 is 1 siderial day as a fraction of 1 Earth day (24 hours)
            ' 0.0657098244 is also 1 siderial day - as a fraction of 24 hours
            ' 36,525 is days in a Julian century (i.e. 1 julian year is exactly 365.25 days)


            ' The G value of 6.697374558 used here is for Greenwich hour angle of the Sun at Epoch2000 - need to adjust for the specified date
            lT = julianIAUDate(pDy, pMn, pYr)
            lT = (lT / 36525.0) - 1
            lR0 = lT * (0.0513366 + lT * (0.00002586222 - lT * 0.000000001722))
            lR1 = 6.697374558 + 2400.0 * (lT - ((pYr - 2000.0) / 100.0))
            lT0 = util.normalise0to24(lR0 + lR1)

            ' Calc UT -  the local time with any Daylight Saving (0 or 1) and Time Zone (-12 to +12) factored in
            pUT = pLocalTm - pDs - pTz
            util.normalise0to24withDay(pUT, lNxtPrvDayIndicator)

            ' Calc GMST
            If lNxtPrvDayIndicator = 1 Then
                lT0 += 0.0657098244
            End If

            If lNxtPrvDayIndicator = -1 Then
                lT0 -= 0.0657098244
            End If

            pGST = (pUT * 1.002737908) + lT0
            pGST = util.normalise0to24(pGST)


            ' Calc LST - GST plus longitude (which may be negative)
            pLST = pGST + (pLongitude / 15.0)
            pLST = util.normalise0to24(pLST)


        Catch ex As Exception
            util.Err(ex.Message, System.Reflection.MethodBase.GetCurrentMethod.Name)
        End Try

    End Sub



    ' Convert calendar date to The IAU accepted modified definition of Julian Date.
    ' Note: There are numerous variations (different epochs) for Julian date - this calculates the 
    ' IAU modified Julian date.
    Private Function julianIAUDate(ByVal pDay As Double, ' This may inlcude the time as a fraction of the day
                                   ByVal pMonth As Integer,
                                   ByVal pYear As Integer) As Double
        Try

            ' One simple formulae for JD published comprehensively is:
            ' 1. Adjust if Jan or Feb - subtract 1 from year and add 12 to month 
            ' 2. A = Y / 100
            ' 3. B = A / 4
            ' 4. C = 2 - A + B
            ' 5. E = 365.25x(Y+4716)
            ' 6. F = 30.6001x(M+1)
            ' 7. JD = C + D + E + F - 1524.5
            ' Some interesting magic numbers, for example:
            ' - 30.6001 is a well known date constant value used for the average days per month without Jan and Feb.
            ' - 4716 IS 4716BC - the epoch being used.
            ' - 365.25 is days in a year (hence 1 extra day every four years, a leap year, for the 0.25), and 1 julian
            '   year is defined as 365.25 days.
            ' - 1524.5 The epoch used for IAU JD starts at midday (0.5 of a day) and hence if the pDay 
            ' If pDay is passed in as an integer value the returned JD will always end as .5  e.g. 20th August 2020 is IAU JD 44061.5
            ' as the modified julian date starts at 12:00 noon.

            ' Adjustment required if Jan or Feb
            If pMonth <= 2 Then
                pYear -= 1
                pMonth += 12
            End If

            ' Caluclate the long Julian Date
            Dim A As Double = CDbl(Math.Floor(pYear / 100.0))
            Dim B As Double = 2 - A + Math.Floor(A / 4)
            Dim JD As Double = Math.Floor(365.25 * (pYear + 4716)) + Math.Floor(30.6001 * (pMonth + 1)) + pDay + B - 1524.5

            ' To convert the long JD to IAU JD subtract 2415020 (days difference between JD epoch and IAU JD epoch)
            JD -= 2415020

            Return (JD)


        Catch ex As Exception
            util.Err(ex.Message, System.Reflection.MethodBase.GetCurrentMethod.Name)
        End Try
    End Function

    ' Calculate Hour angle (a point on the celestial sphere in the equatorial co-ordinate system).
    ' Hour Angle is LST minus RA normalised to a value between 0 and 24.
    ' Longitude must be passed in as degrees
    Private Function hourAngle(ByVal pRA As Double,
                                ByVal pGST As Double,
                                ByVal pLongitude As Double) As Double
        Try

            ' GST and RA are in hours, Longitude is in degrees and hence must be divided by 15 to reduce to hours to 
            ' keep units matched (as 1 hour = 15 degrees as 24 hours = 360 degrees)
            ' Ensure that the result is always between 0 and 24 hours by calling normalise0to24()

            Return (util.normalise0to24(util.normalise0to24(pGST + (pLongitude / 15.0)) - pRA))


        Catch ex As Exception
            util.Err(ex.Message, System.Reflection.MethodBase.GetCurrentMethod.Name)
        End Try
    End Function


    ' Equatorial to horizontal Conversion.
    ' Horizontal co-ordinates are based upon an Earth based observers poistion (i.e. the Altitude and Azimuth for an observer)
    ' Equatorial co-ordinates are  based upon a fixed position at earth centre and celestial poles.
    ' pAzEquatorial must be passed in as Rads
    ' pDecEquatorial must be passed in as degrees
    ' pLatitude must be passed in as Rads
    Private Sub equatorialToHorizontal(ByVal pAzEquatorial As Double,
                      ByVal pDecEquatorial As Double,
                      ByVal pLatitude As Double,
                      ByRef pAzHorizontal As Double,
                      ByRef pAltHorizontal As Double)
        Try

            ' Use the following cosine rule to get Altitude: sin(a) = sin(δ) sin(φ) + cos(δ) cos(φ) cos(H)
            ' Use the following cosine rule to get azimuth: cos(A) = { sin(δ) - sin(φ) sin(a) } / cos(φ) cos(a)


            pAltHorizontal = Math.Asin((Math.Sin(pDecEquatorial) * Math.Sin(pLatitude)) + (Math.Cos(pDecEquatorial) * Math.Cos(pLatitude) * Math.Cos(pAzEquatorial)))
            pAzHorizontal = Math.Acos((Math.Sin(pDecEquatorial) - (Math.Sin(pLatitude) * Math.Sin(pAltHorizontal))) / (Math.Cos(pLatitude) * Math.Cos(pAltHorizontal)))

            ' Check if equatorial azimuth is > 180 degrees (sin is negative) and correct for horizontal azimuth
            If Math.Sin(pAzEquatorial) > 0 Then
                pAzHorizontal = 2 * Math.PI - pAzHorizontal
            End If


        Catch ex As Exception
            util.Err(ex.Message, System.Reflection.MethodBase.GetCurrentMethod.Name)
        End Try

    End Sub

End Class
